const express = require('express');
const cors = require('cors');
const bodyParser = require('body-parser');
const ping = require('ping');
const nodemailer = require('nodemailer');
const fs = require('fs'); // Import the file system module

const app = express();
const PORT = process.env.PORT || 5000;

app.use(cors());
app.use(bodyParser.json());
//app.use(cors({ origin: "https://ping.onlinefashionbd.com" }));

const DATA_FILE = 'ips.json'; // File to store IP data

// Load IP data from file (if it exists)
let ips = [];
if (fs.existsSync(DATA_FILE)) {
    const data = fs.readFileSync(DATA_FILE, 'utf8');
    ips = JSON.parse(data);
}

// Save IP data to file
const saveData = () => {
    fs.writeFileSync(DATA_FILE, JSON.stringify(ips, null, 2)); // Pretty-print JSON
    console.log('Data saved to file.');
};

// Email configuration
const transporter = nodemailer.createTransport({
    service: 'gmail', // Use your email service
    auth: {
        user: 'reajulislamnizhumgmail.com', // Your email address
        pass: '', // Your email password
    },
});

const sendEmailAlert = (ip, status, title, location, contactPersonName, contactPersonNumber, lastStatusChange, duration) => {
    const formattedLastStatusChange = lastStatusChange.toLocaleString(); // Format the timestamp

    const mailOptions = {
        from: '"Network Monitoring Tool" <reajulislamnizhum@gmail.com>',
        to: 'nizhum@tnabd.com', // Recipient email address
        subject: `IP Status Alert: ${ip} is ${status}`,
        text: `Dear Admin,

            The IP address ${ip} is now ${status}.

            Details:
            -> Title: ${title}
            -> Location: ${location}
            -> Contact Person: ${contactPersonName} 
            -> Contact Number: ${contactPersonNumber}
            -> Last Status Change: ${formattedLastStatusChange}
            -> Duration: ${duration}

            Best Regards,
            Network Monitoring Tool`,
    };

    transporter.sendMail(mailOptions, (error, info) => {
        if (error) {
            console.error('Error sending email:', error);
        } else {
            console.log('Email sent:', info.response);
        }
    });
};

app.post('/ips', (req, res) => {
    const { ip, title, location, contactPersonName, contactPersonNumber } = req.body;
    if (!ips.some((item) => item.ip === ip)) {
        const now = new Date();
        ips.push({
            ip,
            title: title || 'Not Available',
            location: location || 'Not Available',
            contactPersonName: contactPersonName || 'Not Available',
            contactPersonNumber: contactPersonNumber || 'Not Available',
            status: '-',
            totalUptime: 0,
            totalDowntime: 0,
            lastFailedOn: null,
            lastSucceedOn: null,
            lastStatusChange: null,
            startTime: now,
            history: [],
        });
        saveData(); // Save data to file
    }

    console.log('Updated IP List:', ips);
    res.json(ips);
});

app.post('/ips/bulk', (req, res) => {
    const newIps = req.body;

    // Validate the input
    if (!Array.isArray(newIps)) {
        return res.status(400).json({ error: 'Invalid input: Expected an array of IPs' });
    }

    // Add each IP to the ips array
    newIps.forEach((ipData) => {
        const { ip, title, location, contactPersonName, contactPersonNumber } = ipData;

        // Check if the IP already exists
        if (!ips.some((item) => item.ip === ip)) {
            const now = new Date();
            ips.push({
                ip,
                title: title || 'Not Available',
                location: location || 'Not Available',
                contactPersonName: contactPersonName || 'Not Available',
                contactPersonNumber: contactPersonNumber || 'Not Available',
                status: '-',
                totalUptime: 0,
                totalDowntime: 0,
                lastFailedOn: null,
                lastSucceedOn: null,
                lastStatusChange: null,
                startTime: now,
                history: [],
            });
        }
    });

    saveData(); // Save data to file
    console.log('Updated IP List:', ips);
    res.json(ips);
});

app.get('/ips', (req, res) => {
    const formattedIps = ips.map((item) => ({
        ...item,
        totalUptimeFormatted: formatTime(item.totalUptime),
        totalDowntimeFormatted: formatTime(item.totalDowntime),
    }));
    res.json(formattedIps);
});

const getHistoryForIp = (ip) => {
    const ipEntry = ips.find((item) => item.ip === ip);
    return ipEntry ? ipEntry.history : [];
};

app.get('/ips/:ip/history', (req, res) => {
    const ip = req.params.ip;
    console.log(`Fetching history for IP: ${ip}`);

    const history = getHistoryForIp(ip);

    if (history.length > 0) {
        res.json(history);
    } else {
        res.status(404).json({ error: 'No history found for this IP' });
    }
});

const formatTime = (seconds) => {
    const minutes = Math.floor(seconds / 60);
    const hours = Math.floor(minutes / 60);
    return `${hours}h ${minutes % 60}m`;
};

app.delete('/ips/:ip', (req, res) => {
    const { ip } = req.params;
    ips = ips.filter((item) => item.ip !== ip);
    saveData(); // Save data to file
    res.json(ips);
});

app.put('/ips/:ip', (req, res) => {
    const { ip } = req.params;
    const { title, location, contactPersonName, contactPersonNumber } = req.body;

    let ipEntry = ips.find((item) => item.ip === ip);

    if (!ipEntry) {
        return res.status(404).json({ success: false, message: 'IP Address not found' });
    }

    ipEntry.title = title || ipEntry.title;
    ipEntry.location = location || ipEntry.location;
    ipEntry.contactPersonName = contactPersonName || ipEntry.contactPersonName;
    ipEntry.contactPersonNumber = contactPersonNumber || ipEntry.contactPersonNumber;

    saveData(); // Save data to file
    console.log('Updated IP:', ipEntry);

    res.json({ success: true, data: ips });
});

const updateIpStatus = async () => {
    for (const item of ips) {
        try {
            const pingPromise = ping.promise.probe(item.ip, { timeout: 15 });
            const timeoutPromise = new Promise((resolve) => {
                setTimeout(() => resolve({ alive: false }), 15000);
            });

            const result = await Promise.race([pingPromise, timeoutPromise]);
            const now = new Date();
            const newStatus = result.alive ? 'online' : 'offline';
            const duration = (now - item.history) / 1000;
            if (item.status !== newStatus) {
                if (!item.pendingStatus || item.pendingStatus !== newStatus) {
                    // First detection of a potential status change
                    item.pendingStatus = newStatus;
                    item.pendingStatusTime = now;
                } else if ((now - item.pendingStatusTime) / 1000 >= 15) {
                    // If status change persists for 15 seconds, apply it
                    if (item.history.length > 0) {
                        const lastHistory = item.history[item.history.length - 1];
                        lastHistory.endedAt = now;
                        lastHistory.duration = (now - new Date(lastHistory.startedAt)) / 1000;
                    }

                    item.history.push({ status: newStatus, startedAt: now, endedAt: null, duration: null });

                    item.lastStatusChange = now;
                    item.status = newStatus;

                    if (newStatus === 'online') {
                        item.lastSucceedOn = now;
                    } else {
                        item.lastFailedOn = now;
                    }

                    sendEmailAlert(
                        item.ip,
                        newStatus,
                        item.title,
                        item.location,
                        item.contactPersonName,
                        item.contactPersonNumber,
                        item.lastStatusChange,
                        duration
                    );
                }
            } else {
                // Reset pending status if it reverts
                item.pendingStatus = null;
                item.pendingStatusTime = null;
            }

            if (newStatus === 'online') {
                item.totalUptime += 5;
            } else {
                item.totalDowntime += 5;
            }
        } catch (error) {
            console.error(`Error pinging ${item.ip}:`, error);
        }
    }
    saveData(); // Save data to file after updating statuses
};

setInterval(updateIpStatus, 5000);

app.listen(PORT, () => {
    console.log(`Server running on http://localhost:${PORT}`);
});